/*
 * Copyright (C) 1996 Be, Inc.  All Rights Reserved
 * Marginally modified by Simon Thornington
 * Release: 0.1a
 * 15/04/97
 * 
 * thorns@cs.mcgill.ca
 */
#include <netconfig.h>
#include <stdio.h>
#include <stdlib.h>
#include <socket.h>
#include <netdb.h>
#include <ether_driver.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <Handler.h>
#include <TextControl.h>
#include <Window.h>
#include <Button.h>
#include <PopUpMenu.h>
#include <MenuField.h>
#include <MenuBar.h>
#include <MenuItem.h>
#include <TextView.h>

static const char DRIVER_NAME[] = "/dev/hp100";			/* our device-driver */
static const char TITLE[] = "HP 10/100VG J2585B PCI";	/* our pretty name */

extern "C" void	_kdprintf_(const char *format, ...);
extern "C" bool _kset_dprintf_enabled_(bool); 

#define set_ddprintf _kset_dprintf_enabled_ 
#define ddprintf _kdprintf_		/* debugging only - turned on */

typedef unsigned long ip_address_t;	/* an IP address */


/*
 * Message codes
 */
enum control_codes {
	IPHIT = 0x1000,
	NMHIT,
	OKHIT,
	CANCELHIT
};

/*
 *  Hard-code some UI characteristics
 */
#define BUTTON_HEIGHT 18

static const rgb_color BACKGROUND_COLOR = { 216, 216, 216, 0 };
static const rgb_color WHITE_COLOR = { 255, 255, 255, 0 };
static const rgb_color BLACK_COLOR = { 0, 0, 0, 0 };

/*
 * Our window class
 */
class HWindow : public BWindow {
	sem_id sem;
	int status;
public:
	HWindow(BRect frame,
			const char *title, 
			window_type type,
			ulong flags,
			ulong workspace = B_CURRENT_WORKSPACE)
	: BWindow(frame, title, type,  flags, workspace) {
		sem = create_sem(0, "hp window");
		status = 0;
	}
	~HWindow(void) {
		delete_sem(sem);
	}
	int Wait(void) {
		if ((acquire_sem_etc(sem,1,B_TIMEOUT,20000000)) == B_TIMED_OUT)
			return(-1);
		else
			return (status);
	}
	void MessageReceived(BMessage *msg) {
		switch (msg->what) {
		case IPHIT:
			break;
		case NMHIT:
			break;
		case OKHIT:
			release_sem(sem);
			break;
		case CANCELHIT:
			status = -1;
			release_sem(sem);
			break;
		}
	}
	bool FilterKeyDown(ulong *key, BView **targetView) {
		if (*key == B_ESCAPE) {
			status = -1;
			release_sem(sem);
		}
		return (inherited::FilterKeyDown(key, targetView));
	}
};

/*
 * compute a text rect, given a frame
 */
static BRect 
textrect(BRect frame)
{
 	frame.InsetBy(2, 2);
	frame.OffsetTo(0, 0);
	return (frame);
}

/*
 * Initialize the text control
 */
static void
textcontrol_init(
				 BTextControl *tc
				 )
{
	BTextView *tv;

	tv = (BTextView *)tc->ChildAt(0);
	tv->SetViewColor(WHITE_COLOR);
	tv->SetHighColor(BLACK_COLOR);
	tv->SetLowColor(WHITE_COLOR);
}

/*
 * Inititialize the title view 
 */
static void
title_init(
		  BTextView *view,
		  const char *text
		  )
{
	
	view->MakeEditable(FALSE);
	view->MakeSelectable(FALSE);
	view->SetViewColor(BACKGROUND_COLOR);
	view->SetHighColor(BLACK_COLOR);
	view->SetLowColor(BACKGROUND_COLOR);
	view->SetText(text);
}


/*
 * Set the font for the menu field
 */
static void
set_font(
		 BPopUpMenu *menu,
		 BMenuField *field
		 )
{
	field->SetFontName("Erich");
	field->MenuBar()->SetFontName("Erich");
	menu->SetFontName("Erich");
}

/*
 * Here are the entry points: the pragma causes these entry points to 
 * exported from the add-on.
 */

#pragma export on

/*
 * configure: put up a window to let the user set the 
 * IP address, netmask and ISA settings for this card.
 *
 * Returns a non-negative number if the new settings should be accepted, or
 * a negative number indicating the user canceled the operation.
 *
 */
extern "C" long
configure(
		  ip_address_t *ipaddressp,
		  ip_address_t *netmaskp,
		  char *driver_name,
		  unsigned *nprivate,
		  nc_private_t *private_items
		  )
{
	HWindow *win;
	BTextControl *ipedit;
	BTextControl *nmedit;
	BButton *okbut;
	BButton *cancelbut;
	BView *background;
	BRect test;
	struct in_addr ip;
	struct in_addr nm;
	char ascii_ip[16];
	char ascii_nm[16];
	BMenuItem *mitem;
	rgb_color hi;
	rgb_color lo;
	BTextView *title;
	int status;
	
	static BRect WINRECT(		140.0,	140.0,		360.0,		300.0);
	static BRect TITLEFRAME(	10.0,		10.0,		200.0,		25.0);
	static BRect IPFRAME(		10.0,		30.0,		200.0,		55.0);
	static BRect NMFRAME(		10.0,		70.0,		200.0,		85.0);
	static BRect CANCELFRAME(	10.0,		120.0,	90.0,			120.0 + BUTTON_HEIGHT);
	static BRect OKFRAME(		120.0,	120.0,	200.0,		120.0 + BUTTON_HEIGHT);

	ip.s_addr = *ipaddressp;
	nm.s_addr = *netmaskp;
	strcpy(ascii_ip, inet_ntoa(ip));
	strcpy(ascii_nm, inet_ntoa(nm));
	win = new HWindow(WINRECT, TITLE, B_MODAL_WINDOW, 
					  B_NOT_RESIZABLE | B_NOT_ZOOMABLE | B_NOT_CLOSABLE);

	title = new BTextView(TITLEFRAME, "title", textrect(TITLEFRAME), 
						  B_FOLLOW_NONE, B_WILL_DRAW);
	ipedit = new BTextControl(IPFRAME, "", "IP Address", 
							  ascii_ip, new BMessage(IPHIT));
	ipedit->SetFontName("Erich");
	nmedit = new BTextControl(NMFRAME, "", "Netmask", ascii_nm, 
							  new BMessage(NMHIT));
	nmedit->SetFontName("Erich");
	okbut = new BButton(OKFRAME, "", "Set", new BMessage(OKHIT));
	cancelbut = new BButton(CANCELFRAME, "", "Cancel", new BMessage(CANCELHIT));

	background = new BView(win->Bounds(), "", B_FOLLOW_ALL, B_WILL_DRAW);

	win->Lock();
	win->AddChild(background);
	background->SetViewColor(BACKGROUND_COLOR);

	background->AddChild(title);
	background->AddChild(ipedit);
	background->AddChild(nmedit);
	background->AddChild(cancelbut);
	background->AddChild(okbut);
	okbut->MakeDefault(TRUE);

	title_init(title, TITLE);
	textcontrol_init(ipedit);
	textcontrol_init(nmedit);
	
	ipedit->MakeFocus(TRUE);
	win->Unlock();
	win->Show();
	status = win->Wait();
	if (status >= 0) {
		*ipaddressp = inet_addr(ipedit->Text());
		*netmaskp = inet_addr(nmedit->Text());
		strcpy(driver_name, DRIVER_NAME);
	}

	win->Quit();

	return (status);
}

/*
 * Open the driver, and initialize it using the configuration data 
 * ISA settings. Return the file descriptor associated with the
 * device-driver, or -1 upon error.
 *
 */
extern "C" int
driver_open(const char *device)
{
	char devicelink[NC_MAXVALLEN+1];
	int fd;

	if (!netconfig_find(device, "DEVICELINK", devicelink)) {
		return (-1);
	}
	fd = open(devicelink, O_RDWR, 0);
	ddprintf("hp100: opened %s with fd %d\n", devicelink,fd);
	if (fd < 0) {
		return (-1);
	}
	if (ioctl(fd, ETHER_INIT, NULL, 0) < 0) {
		close(fd);
		return (-1);
	}
	return (fd);
}

/*
 * Return the pretty name for this driver
 */
extern "C" void
get_pretty_name(char *pretty_name)
{
	strcpy(pretty_name, TITLE);
}

#pragma export reset
